function V = KMSDA(K, gnd, gnd_sub, options)

%
% function V = KMSDA(K, gnd, gnd_sub, opts)
%
% KMSDA: Compute KMSDA projection matrix using an efficient eigenvalue
% decomposition implementation [1,2,3].
%
% 
% IN
%
% K: kernel Gram matrix - N x N
%
% gnd: class ground truth labels - N x 1
%
% gnd_sub: subclass ground truth labels - N x 1
%
% options:    - options field for future use
%
%
% OUT
%
% V: projection matrix - D x N (D << N)
%
%
% Related references:
%
% 1. N. Gkalelis, V. Mezaris, I. Kompatsiaris, T. Stathaki, "Mixture
% subclass discriminant analysis link to restricted Gaussian model and 
% other generalizations", IEEE Transactions on Neural Networks and
% Learning Systems, vol. 24, no. 1, pp. 8-21, January 2013. 
%
% 2. M. Zhu and A.M. Martinez, "Subclass Discriminant Analysis", IEEE 
% Transactions on Pattern Analysis and Machine Intelligence, Vol. 28,
% No. 8, pp. 1274-1286, 2006
%
% 3. D. Cai, X. He, and J. Han. Speed up kernel discriminant analysis. The
% VLDB Journal, 20(1):21{33, Feb. 2011.
%
%
% Author: Nikolaos Gkalelis - CERTH-ITI
% Email: gkalelis@iti.gr
%
% Created 01 Dec 2013.
%


% fd = 1;
% fprintf(fd , 'KMSDA>> Entering\n');

smallVal = 1e-6;

Clbl = unique(gnd);
C = length(Clbl);
N = size(K,1);

K = max(K,K');
K_orig = K;

% centering
km_vec = sum(K,2);
Km = repmat(km_vec./N,1,N);
K = K - Km - Km' + sum(km_vec)/(N^2);
K = max(K,K');
clear Km km_vec;

% eigen decomposition of kernel matrix
[Ur,Sr] = eig(K);
Sr = diag(Sr);

maxEigValue = max(abs(Sr));
eigValIdx = find(abs(Sr)/maxEigValue < smallVal);
if length(eigValIdx) < 1
    [dump,eigValIdx] = min(Sr);
    clear dump;
end
Sr(eigValIdx) = [];
Ur(:,eigValIdx) = [];

% compute Sr factor using Ur
Hlbl = unique([gnd, gnd_sub], 'rows');
H = size(Hlbl,1);
D = H - 1; % projection space dimensionality

Hi = zeros(C,1); % number of subclasses per class
for i=1:C
    Hi(i) = sum(Hlbl(:,1) == Clbl(i));
end

% compute subclass means
for i = 1:C
    for j=1:Hi(i)
        subClassMean(i).meanvec(j,:) = mean(Ur(gnd==i & gnd_sub==j, :) );
        subClassMean(i).Ni(j) = sum(gnd==i & gnd_sub==j);
    end
end

% compute total number of combinations of subclass means
Nh = 0;
for i=1:C-1
    for j=i+1:C
        Nh = Nh + Hi(i)*Hi(j);
    end
end

% compute factor of Ur^T B Ur = Hb Hb^T
Hb = zeros(Nh,size(Ur,2));
t=0;
for i = 1:C-1
    for j=1:Hi(i)
        for k=i+1:C
            for l=1:Hi(k)
                t = t+1;
                Hb(t,:) = sqrt(subClassMean(i).Ni(j)*subClassMean(k).Ni(l))...
                    *(subClassMean(i).meanvec(j,:) - subClassMean(k).meanvec(l,:));
            end
        end
    end
end

if t ~= Nh
    error('KMSDA>>> Error in total number of subclass means !');
end

% fast computation of eigen vectors of Ur^T B Ur
[dumpVec,G,W] = svd(Hb,'econ');
clear dumpVec;
G = diag(G);

% keep only required eigenvectors
if length(G) > D
    G = G(1:D);
    W = W(:,1:D);
end

% compute eigenvectors of original problem
W =  (Ur.*repmat((Sr.^-1)',N,1))*W;

% normalize to unitary matrix
tmpNorm = sqrt(sum((W'*K_orig).*W',2));
V = W./repmat(tmpNorm',size(W,1),1);

% fprintf(fd , 'KMSDA>> Exiting\n');

