
function [c_id DJmin] = selectClassToRePartitionPost( fea, gnd,  gnd_subclass, distType, Nmn)
%
% function [c_id DJmin] = selectClassToRePartitionPost( fea, gnd,  gnd_subclass, distType, Nmn)
%
% Select class to repartition using a nongaussianity measure [1,2]. In this
% implementation the negentropy increment is used [3,4].
% That is, repartition the class that will yield the most Gaussian-like
% subclass structure.
%
% IN
%
% fea: feature vector observations - N x F
%
% gnd: ground truth labels - N x 1
%
% gnd_subclass: ground truth labels at subclass labels - together with gnd 
% provide a 2D subclass label for each observation - N x 1
%
% distType: similarity (or disimilarity) measure to be used (Euclidean
% distance or cosine similarity)
%
% OUT
%
% c_id: index of class to repartition
%
% DJmin: the negentropy increment value for the new partition
%
% Related references:
%
% 1. N. Gkalelis, V. Mezaris, I. Kompatsiaris, "Mixture subclass discriminant
% analysis", IEEE Signal Processing Letters, vol. 18, no. 5, pp. 319-322, 
% May 2011
%
% 2. N. Gkalelis, V. Mezaris, I. Kompatsiaris, T. Stathaki, "Mixture subclass 
% discriminant analysis link to restricted Gaussian model and other 
% generalizations", IEEE Transactions on Neural Networks and Learning 
% Systems, vol. 24, no. 1, pp. 8-21, January 2013.
%
% 3. Luis F. Lago-Fernandez, Fernando J. Corbacho: Using the Negentropy
% Increment to Determine the Number of Clusters. IWANN (1) 2009: 448-455
%
% 4. Luis F. Lago-Fernandez, Fernando J. Corbacho: Normality-based
% validation for crisp clustering. Pattern Recognition 43(3): 782-795
% (2010)
%
%
% Author: Nikolaos Gkalelis - CERTH-ITI
% Email: gkalelis@iti.gr
%
% Created 01 Aug 2013.
%


% fd = 1;
% fprintf(fd , 'selectClassToRePartitionPost>> Entering\n');


%% data details
classLbls = unique(gnd);
C = length(classLbls); % number of classes

%% repartition each class separetaly and compute overall nongaussianity
DJ = zeros(1,C); % 
for i = 1:C
    gnd_subclass_tmp = gnd_subclass; % initialize ground truth
    Hi = length(unique( gnd_subclass_tmp( gnd == i, 1 ))); % current number of subclasses of Class to Re-Partition
    [ fea_sorted_tmp gnd_tmp Ni stopPartitioningThisClass] = incrementSubclassesOfClass( fea( gnd == i, :), Hi+1, distType, Nmn); % check Hi = Hi+1
    
    if stopPartitioningThisClass == true
        DJ(i) = realmax; % do not select this class !!!
        continue;
    end
    
    fea( gnd == i, :) = fea_sorted_tmp; % sort class observations according to the new subclass labelling
    gnd_subclass_tmp(gnd == i, :) = gnd_tmp; % do the same for ground truth data
    
    DJ(i) = negEntropyIncrement(fea, gnd, gnd_subclass_tmp); % compute total negentropy where Hi = Hi+1
end

if sum(DJ == realmax) == C
    error('selectClassToRePartitionPost>> Minimum number of observations has reached in all subclasses');
end

%% select class that yields minimum nongaussianity
% That is, yields the most Gaussian-like subclass structure
[DJmin c_id] = min(DJ);

% fprintf(fd , 'selectClassToRePartitionPost>> Exiting\n');


