
function res = test_kmsda(prm)
%
%function res = test_kmsda(prm)
%
% Classify a test set using KMSDA and the nearest neighbour (NN) classifier
%
%
% IN
%
% prm : input parameter structure with the following fields:
%
%  fea_Test: test feature vectors Ntst x F
%
%  gnd_Test: ground truth labels of test dataset (used for computing
%  correct recognition rate (CCR)) - Ntst x 1
%
%  fea_Train: training feature vectors (used from NN classifier) Ntrn x F
%
%  gnd_Train: ground truth labels of training dataset Ntrn x 1
%
%  fea_Mean: mean sample vector 1 x F comuted using the training dataset
% (empty [] if unity norm or no normalization is used)
%
%  W: projection matrix (has been computed using the training dataset)
%
%  normType: method to use for normalizing the observations, i.e. zero mean 
%  sample, unity norm, or no normalization at all
%
%  distType: similarity (or disimilarity) measure to use for comparing
%  observations, i.e. Euclidean distance, or cosine similarity.
%
% OUT
%
% res : output parameter structure with the following fields:
%
%  CCR: CCR computed using NN classifier
%  
% Author: Nikolaos Gkalelis - CERTH-ITI
% Email: gkalelis@iti.gr
%
% Created 01 Dec 2013.
%

%% read input parameters
normType = prm.normType;
distType = prm.distType;
kernel_opts = prm.kernel_opts;
fea_Train_Proj = prm.fea_Train_Proj;
fea_Train = prm.fea_Train; % train set
gnd_Train = prm.gnd_Train;
fea_Mean = prm.fea_Mean;
fea_Test = prm.fea_Test; % test set
gnd_Test = prm.gnd_Test;
W = prm.W;
clear prm;

%% trace
logFile = char( strcat('log', '_', 'test_kmsda', '_', normType, '_', distType, '.txt'));
fd = fopen(logFile, 'a');

fprintf(fd , 'test_kmsda>> Entering\n');

%% normalize test set
Nts = size(fea_Test, 1);
if strcmp(normType,'zeroMean')
    fea_Test = fea_Test - repmat(fea_Mean, Nts,1);
elseif strcmp(normType,'unityNorm')
    for n=1:Nts
        fea_Test(n,:) = fea_Test(n,:) ./ max(eps,norm(fea_Test(n,:)));
    end
elseif strcmp(normType,'NoNorm')
end

%% project data
fea_Test_kernel = constructKernel(fea_Test, fea_Train, kernel_opts);
fea_Test_Proj = fea_Test_kernel* W; % project test set

%% recognition
recognized = knnclassify(fea_Test_Proj, fea_Train_Proj, gnd_Train, 1, distType, 'nearest');
res.CCR = sum(gnd_Test == recognized)/length(gnd_Test);

fprintf(fd , 'test_kmsda>> CCR (%d)\n', res.CCR);

fprintf(fd , 'test_kmsda>> Exiting\n');

fclose(fd);

