function [Wmsda, Phi] = cmp_msda_mat(gnd, gnd_sub, options, X)
%
% function [Wmsda, Phi] = cmp_msda_mat(gnd, gnd_sub, options, X)
%
% MSDA: Compute MSDA projection matrix and discriminant analysis (DA)
% stability criterion value [1,2,3]. The efficient eigenvalue decomposition
% implementation described in [4] is used.
%
% 
% IN
%
% X: training feature vectors - N x F
%
% gnd: class ground truth labels - N x 1
%
% gnd_sub: subclass ground truth labels - N x 1
%
% options:    - WithinScatter   1: St
%                               2: Sws
%                               3: Sws + Sbsb
%
%
% OUT
%
% Wmsda: MSDA projection matrix - F x D (D << F)
%
% Phi: DA stability criterion value
%
%
% Related references:
%
% 1. N. Gkalelis, V. Mezaris, I. Kompatsiaris, "Mixture subclass discriminant
% analysis", IEEE Signal Processing Letters, vol. 18, no. 5, pp. 319-322, 
% May 2011
%
% 2. N. Gkalelis, V. Mezaris, I. Kompatsiaris, T. Stathaki, "Mixture subclass 
% discriminant analysis link to restricted Gaussian model and other 
% generalizations", IEEE Transactions on Neural Networks and Learning 
% Systems, vol. 24, no. 1, pp. 8-21, January 2013.
%
% 3. M. Zhu and A.M. Martinez, "Subclass Discriminant Analysis", IEEE 
% Transactions on Pattern Analysis and Machine Intelligence, Vol. 28,
% No. 8, pp. 1274-1286, 2006
%
% 4. Deng Cai, Xiaofei He, Jiawei Han, "SRDA: An Efficient Algorithm for
% Large Scale Discriminant Analysis", IEEE Transactions on Knowledge and
% Data Engineering, 2007.
%
%
% Author: Nikolaos Gkalelis - CERTH-ITI
% Email: gkalelis@iti.gr
%
% Created 01 Aug 2013.
%

%% Initialization
stb_thr = 2000; % threshold for computing DA stability criterion
eps_s = 1e-3; % small number
eps_vs = 1e-12; % very small number

[N,F] = size(X);
if length(gnd) ~= N
    error('cmp_msda_mat>>> gnd and data mismatch!');
end

Clbl = unique(gnd);
C = length(Clbl);
Hlbl = unique([gnd, gnd_sub], 'rows');
H = size(Hlbl,1);
D = H - 1; % projection space dimensionality

Hi = zeros(C,1); % number of subclasses per class
for i=1:C
    Hi(i) = sum(Hlbl(:,1) == Clbl(i));
end

x_m = mean(X,1); % zero mean
X = (X - repmat(x_m,N,1));

% check if Sws is ill-posed
cholp = 0;
if N > F   
    if options.WithinScatter == 1
        Sws = X'*X;
    elseif options.WithinScatter == 2 || options.WithinScatter == 3
        Xs = [];
        for i = 1:C
            for j=1:Hi(i)
                Xij = X(gnd==i & gnd_sub==j, :);
                Nij = size(Xij, 1);
                Xij = Xij - repmat(mean(Xij), Nij, 1);
                Xs = [Xs; Xij];
            end
        end
        Sws = Xs'*Xs; % Sw
    end
    
    Sws = max(Sws,Sws');
    [R,cholp] = chol(Sws);
end

% SSS problem
if F >= N || cholp ~= 0
    usePCA = 1;
    Dpca = N - H;
else
    usePCA = 0;
end

%% use PCA if Sws is ill-posed
if usePCA == 1 % use PCA prestep
    
    if N > F
        Spca = X'*X; % F x F
    else % SSS problem
        Spca = X*X'; % N x N
    end
    
    Spca = max(Spca,Spca');
    
    [evec, Lpca] = eig(Spca);
    Lpca = diag(Lpca);
    clear Spca;
    
    maxEigValue = max(abs(Lpca));
    eigIdx = find(Lpca/maxEigValue < eps_vs);
    Lpca(eigIdx) = [];
    evec(:,eigIdx) = [];
        
    [junk, index] = sort(-Lpca);
    Lpca = Lpca(index);
    
    evec = evec(:, index);
    
    if Dpca < length(Lpca)
        Lpca = Lpca(1:Dpca);
        evec = evec(:,1:Dpca);
    end
    
    Lpca = Lpca.^-.5;
    
    if N > F
        Wpca = evec;
        X = (X*Wpca).*repmat(Lpca',N,1);
    else % SSS problem
        Wpca = (X'*evec).*repmat(Lpca',F,1);
        X = evec; % project to PCA space
    end
    
    clear evec;
end

%% computation of Sbsb
[N,F] = size(X); % PCA may have been applied

% compute subclass means
for i = 1:C
    for j=1:Hi(i)
        subClassMean(i).meanvec(j,:) = mean(X(gnd==i & gnd_sub==j, :) );
        subClassMean(i).Ni(j) = sum(gnd==i & gnd_sub==j);
    end
end

% compute total number of combinations of subclass means
Nh = 0;
for i=1:C-1
    for j=i+1:C
        Nh = Nh + Hi(i)*Hi(j);
    end
end

% compute Sbsb factor (Fbsb)
Fbsb = zeros(Nh,F); % Sbsb = Fbsb'*Fbsb;
t=0;
for i = 1:C-1
    for j=1:Hi(i)
        for k=i+1:C
            for l=1:Hi(k)
                t = t+1;
                Fbsb(t,:) = sqrt(subClassMean(i).Ni(j)*subClassMean(k).Ni(l))...
                    *(subClassMean(i).meanvec(j,:) - subClassMean(k).meanvec(l,:));
            end
        end
    end
end

if t ~= Nh
    error('cmp_msda_mat>>> Error in total number of subclass means !');
end

%% solve eigenvalue problem (if possible using eigs for speed up)
option.disp = 0; % in general for eigs matlab function
if usePCA == 1 % in projection space data are sphered
    [dumpVec,eigval,evec] = svd(Fbsb,'econ');
    
    eigval = diag(eigval);
    eigIdx = find(eigval < eps_s);
    eigval(eigIdx) = [];
    evec(:,eigIdx) = [];

    eigval = eigval.^2;
    evec = Wpca*(repmat(Lpca,1,length(eigval)).*evec);
    Sbsb = Fbsb'*Fbsb; % Sbsb to be used for stability criterion
    
    if options.WithinScatter == 1
        Sws = X'*X;
    elseif options.WithinScatter == 2 || options.WithinScatter == 3
        Xs = [];
        for i = 1:C
            for j=1:Hi(i)
                Xij = X(gnd==i & gnd_sub==j, :);
                Nij = size(Xij, 1);
                Xij = Xij - repmat(mean(Xij), Nij, 1);
                Xs = [Xs; Xij];
            end
        end
        Sws = Xs'*Xs; % Sw
    end
    Sws = max(Sws,Sws');
    
elseif usePCA == 0 % Sws was not ill-posed
    Sbsb = Fbsb'*Fbsb; % Sbsb
    Sbsb = max(Sbsb,Sbsb');

    Dbsb = size(Sbsb,2);
    if D > Dbsb
        D = Dbsb;
    end
    
    % if possible use eigs functions to speed up computations
    if isfield(options,'bEigs')
        if options.bEigs
            bEigs = 1;
        else
            bEigs = 0;
        end
    else
        if (Dbsb > 1000 && D < Dbsb/10) || (Dbsb > 500 && D < Dbsb/20) || ...
                (Dbsb > 250 && D < Dbsb/30) 
            bEigs = 1;
        else
            bEigs = 0;
        end
    end
    
    % use robust Sws
    if options.WithinScatter == 3
        Sws = Sws + Sbsb;
    end
    
    if bEigs % use eigs to speed up computation
        B = Sws;        
        if bChol == 1 % use cholesky factor - the only difference from using regularized Sws
            option.cholB = 1;
            B = R;
        end
        [evec, eigval] = eigs(Sbsb, B, D, 'la', option);
        %eigval = diag(eigval);
        option.cholB = 0; % reset back for next computations with eigs
    else
        [evec, eigval] = eig(Sbsb,Sws);
        eigval = diag(eigval);

        [junk, index] = sort(-eigval);
        eigval = eigval(index);
        evec = evec(:,index);

        if D < size(evec,2)
            evec = evec(:, 1:D);
            %eigval = eigval(1:D);
        end
    end
else
    error('cmp_msda_mat>>> Unknown options for solving eigenvalue problem !');
end

% normalize
Wmsda = zeros(size(evec));
for i = 1:size(evec,2)
    Wmsda(:,i) = evec(:,i)./norm(evec(:,i));
end

%% compute DA stability criterion value
rankSbsb = rank(Sbsb);
[psiA,la] = eigs(Sbsb, rankSbsb, 'LM', option);
clear la;

[psiB,lb] = eig(Sws);
lb = diag(lb)';
[lb,ind] = sort(lb);
lb = fliplr(lb);
ind = fliplr(ind);
psiB = psiB(:,ind);
psiB(:,lb < (sum(lb)/stb_thr)) = [];
L = size(psiB,2);

Phi = 0;
r = max(1, ceil(rankSbsb/2) );
for i=1:r
    Phi = Phi + sum( (psiA(:,i)' * psiB(:,1:min(i,L)) ).^2 );    
end
Phi = Phi/r;


