

function DJ = negEntropyIncrement(fea, gnd, gnd_subclass)
%
% Compute negentropy increment for the specific subclass partition. In case
% that the data covariance matrix is ill-posed PCA is first applied.
%
%
% IN
%
% fea: observations of specific class - N x F
%
% gnd: class ground truth, 1,2,3, ... - N x 1
%
% gnd_subclass: subclass ground truth labels - in combination with gnd
% above provide a 2D subclass label for each observation - N x 1
%
% OUT
%
% DJ: negentropy increment between unpartitioned and partitioned data - 
% computes overall DJ, i.e., subclass partition against unpartitioned data 
%
%
% Remarks:
% The smallest the DJ the most Gaussian like becomes the dataset using
% clustering
% DJ < 0 : negentropy dicreased by the subclass partitioning (set became
% more Gaussian-like). That is, subclassing was beneficial
% DJ > 0: negentropy increased (the set became less Gaussian-like)
%
% Related references:
%
% N. Gkalelis, V. Mezaris, I. Kompatsiaris, "Mixture subclass discriminant
% analysis", IEEE Signal Processing Letters, vol. 18, no. 5, pp. 319-322, 
% May 2011
%
% N. Gkalelis, V. Mezaris, I. Kompatsiaris, T. Stathaki, "Mixture subclass 
% discriminant analysis link to restricted Gaussian model and other 
% generalizations", IEEE Transactions on Neural Networks and Learning 
% Systems, vol. 24, no. 1, pp. 8-21, January 2013.
%
% Luis F. Lago-Fernandez, Fernando J. Corbacho: Using the Negentropy
% Increment to Determine the Number of Clusters. IWANN (1) 2009: 448-455
%
% Luis F. Lago-Fernandez, Fernando J. Corbacho: Normality-based
% validation for crisp clustering. Pattern Recognition 43(3): 782-795
% (2010)
%
% Deng Cai, Xiaofei He, Jiawei Han, and Hong-Jiang Zhang, "Orthogonal
% Laplacianfaces for Face Recognition", in IEEE TIP, 2006.
%
% Author: Nikolaos Gkalelis - CERTH-ITI
% Email: gkalelis@iti.gr
%
% Created in 01 Aug 2013.
%

%% in case of SSS problem perform PCA
[N F] = size(fea); % feature
if N < F
    options=[]; options.ReducedDim=0;
    [v_pca, eigvalue] = PCA(fea);
    fea = fea * v_pca;
end

%% compute overall negentropy (of unpartitioned data)
S0 = cov(fea);
J0 = regLogDet(S0);

%% compute subclass contribution in the negentropy increment
subclassLbls = unique([gnd, gnd_subclass], 'rows');

% initialize
H = size(subclassLbls, 1); % total number of subclasses
Ni = zeros(1,H); % number of observations of i-th subclass
Ji = Ni; % subclass contribution to negentropy increment measure
pi = Ni; % prior probability of an observation falling into i-th subclass
ei = Ni; % subclass contribution to overall 'entropy'

% compute subclass contribution to the negentropy increment measure
for i = 1:H
    fea_i = fea(subclassLbls(i,1) == gnd & subclassLbls(i,2) == gnd_subclass,:);
    Ni(i) = size(fea_i,1);
    Si = cov(fea_i);
    Si = max(Si,Si'); % symmetric

    ldSi = regLogDet(Si);
    
    pi(i) = Ni(i)/N;
    ei(i) = pi(i)*log(pi(i));
    Ji(i) =  pi(i) * ldSi;
end

% compute negentropy increment conceiving each subclass as a separate
% cluster of the overall dataset, and the union of all subclasses as the
% initial unpartitioned data
DJ = (1/2)* (sum(Ji) - J0) - sum(ei);




