

%%
function res = test_msda(prm)
%
%
% Classify a test set using MSDA and the nearest neighbour (NN) classifier
%
%
% IN
%
% prm : input parameter structure with the following fields:
%
%  fea_Test: test feature vectors Ntst x F
%
%  gnd_Test: ground truth labels of test dataset (used for computing
%  correct recognition rate (CCR)) - Ntst x 1
%
%  fea_Train: training feature vectors (used from NN classifier) Ntrn x F
%
%  gnd_Train: ground truth labels of training dataset Ntrn x 1
%
%  fea_Mean: mean sample vector 1 x F comuted using the training dataset
% (empty [] if unity norm normalization is used)
%
%  W: projection matrix (has been computed using the training dataset)
%
%  normType: method to use for normalizing the observations, i.e. zero mean 
%  sample or unity norm
%
%  distType: similarity (or disimilarity) measure to use for comparing
%  observations, i.e. Euclidean distance, or cosine similarity.
%
% OUT
%
% res : output parameter structure with the following fields:
%
%  CCR: CCR computed using NN classifier
%  
% Author: Nikolaos Gkalelis - CERTH-ITI
% Email: gkalelis@iti.gr
%
% Created 01 Aug 2013.
%

%% read input parameters
normType = prm.normType;
distType = prm.distType;
WithinScatter = prm.WithinScatter;
fea_Train = prm.fea_Train;
gnd_Train = prm.gnd_Train;
fea_Mean = prm.fea_Mean;
fea_Test = prm.fea_Test; % test set
gnd_Test = prm.gnd_Test;
W = prm.W;
clear prm;

%% trace
logFile = char( strcat('log', '_', 'test_msda', '_', normType, '_', distType, '_', int2str(WithinScatter), '.txt'));
fd = fopen(logFile, 'a');

fprintf(fd , 'test_msda>> Entering\n');

%% normalize
Ntr = size(fea_Train, 1);
Nts = size(fea_Test, 1);
if strcmp(normType,'zeroMean')
    fea_Train = fea_Train - repmat(fea_Mean, Ntr,1);
    fea_Test = fea_Test - repmat(fea_Mean, Nts,1);
elseif strcmp(normType,'unityNorm')
    for n=1:Ntr
        fea_Train(n,:) = fea_Train(n,:) ./ max(eps,norm(fea_Train(n,:)));
    end
    for n=1:Nts
        fea_Test(n,:) = fea_Test(n,:) ./ max(eps,norm(fea_Test(n,:)));
    end
elseif strcmp(normType,'NoNorm')
end

%% project in MSDA subspace and classify using NN classifier
fea_Train_Proj = fea_Train * W; % project train set
fea_Test_Proj = fea_Test * W; % project test set
recognized = knnclassify(fea_Test_Proj, fea_Train_Proj, gnd_Train, 1, distType, 'nearest');
res.CCR = sum(gnd_Test == recognized)/length(gnd_Test);

fprintf(fd , 'test_msda>> CCR (%d)\n', res.CCR);

fprintf(fd , 'test_msda>> Exiting\n');

fclose(fd);

